import { json } from '@sveltejs/kit';
import bcrypt from 'bcrypt';
import jwt from 'jsonwebtoken';
import { db } from '$lib/server/db.js';

const JWT_SECRET = process.env.JWT_SECRET || process.env.SESSION_SECRET || 'siakad-dev-secret-ganti-di-production';
const COOKIE_NAME = 'siakad_token';
const COOKIE_MAX_AGE = 60 * 60 * 24; // 1 hari

/** @type {import('./$types').RequestHandler} */
export async function POST({ request }) {
  try {
    const { email, password } = await request.json();

    if (!email || !password) {
      return json(
        { error: 'Email dan password harus diisi' },
        { status: 400 }
      );
    }

    const trimmedEmail = email.trim();
    const trimmedPassword = password.trim();

    /* ===========================
       1️⃣ CEK USER: EMAIL ATAU NIS (siswa) ATAU NIP (guru) – data yang sudah ada di MySQL bisa login
       =========================== */
    let [users] = await db.query(
      'SELECT * FROM users WHERE email = ?',
      [trimmedEmail]
    );

    if (users.length === 0) {
      const [students] = await db.query(
        'SELECT email FROM students WHERE nis = ? LIMIT 1',
        [trimmedEmail]
      );
      if (students.length > 0) {
        [users] = await db.query(
          'SELECT * FROM users WHERE email = ?',
          [students[0].email]
        );
      }
    }

    if (users.length === 0) {
      const [teachers] = await db.query(
        'SELECT email FROM teachers WHERE nip = ? LIMIT 1',
        [trimmedEmail]
      );
      if (teachers.length > 0) {
        [users] = await db.query(
          'SELECT * FROM users WHERE email = ?',
          [teachers[0].email]
        );
      }
    }

    if (users.length === 0) {
      return json(
        { error: 'Email / NIS / NIP tidak terdaftar' },
        { status: 401 }
      );
    }

    const user = users[0];

    /* ===========================
       2️⃣ CEK PASSWORD: users.password (bcrypt/plain) ATAU NIS/NIP dari tabel students/teachers
       =========================== */
    const storedPassword = user.password && String(user.password);
    const isBcryptHash = storedPassword && (storedPassword.startsWith('$2a$') || storedPassword.startsWith('$2b$') || storedPassword.startsWith('$2y$'));

    let isPasswordValid = false;
    if (isBcryptHash) {
      isPasswordValid = await bcrypt.compare(trimmedPassword, storedPassword);
    } else if (storedPassword) {
      // Data lama: password plain text di users
      isPasswordValid = String(storedPassword).trim() === trimmedPassword;
    }

    // Fallback: siswa → password = NIS (dari tabel students), guru → password = NIP (dari tabel teachers)
    // Agar data yang sudah ada di MySQL tetap bisa login pakai NIS/NIP yang tertera
    if (!isPasswordValid && user.role === 'siswa') {
      const [studentRows] = await db.query(
        'SELECT nis FROM students WHERE email = ? LIMIT 1',
        [user.email]
      );
      if (studentRows.length > 0 && studentRows[0].nis != null) {
        const nisFromDb = String(studentRows[0].nis).trim();
        if (nisFromDb === trimmedPassword) isPasswordValid = true;
      }
    }
    if (!isPasswordValid && user.role === 'guru') {
      const [teacherRows] = await db.query(
        'SELECT nip FROM teachers WHERE email = ? LIMIT 1',
        [user.email]
      );
      if (teacherRows.length > 0 && teacherRows[0].nip != null) {
        const nipFromDb = String(teacherRows[0].nip).trim();
        if (nipFromDb === trimmedPassword) isPasswordValid = true;
      }
    }

    if (!isPasswordValid) {
      return json(
        { error: 'Password salah' },
        { status: 401 }
      );
    }

    /* ===========================
       3️⃣ CEK STATUS (WAJIB)
       =========================== */
    if (user.status !== 'aktif') {
      return json(
        { error: 'Akun belum aktif' },
        { status: 403 }
      );
    }

    /* ===========================
       4️⃣ LOGIN SUKSES – issue JWT & set cookie
       =========================== */
    const tokenPayload = { userId: user.id, role: user.role };
    const token = jwt.sign(tokenPayload, JWT_SECRET, { expiresIn: '24h' });

    const isProd = process.env.NODE_ENV === 'production';
    const cookieValue = `${COOKIE_NAME}=${token}; Path=/; HttpOnly; SameSite=Lax; Max-Age=${COOKIE_MAX_AGE}${isProd ? '; Secure' : ''}`;

    return json(
      {
        success: true,
        user: {
          id: user.id,
          fullName: user.fullName,
          email: user.email,
          phone: user.phone,
          role: user.role,
          major: user.major,
          status: user.status
        }
      },
      {
        headers: {
          'Set-Cookie': cookieValue
        }
      }
    );

  } catch (error) {
    console.error('❌ Login API error:', error);
    return json(
      { error: 'Terjadi kesalahan pada server' },
      { status: 500 }
    );
  }
}
