import { json } from '@sveltejs/kit';
import bcrypt from 'bcrypt';
import { db } from '$lib/server/db.js';

/** @type {import('./$types').RequestHandler} */
export async function GET() {
  try {
    const [users] = await db.query('SELECT id, fullName, email, phone, role, major, status, createdAt FROM users ORDER BY role, fullName');
    return json({ success: true, data: users });
  } catch (error) {
    console.error('Get users error:', error);
    return json({ error: 'Gagal mengambil data users' }, { status: 500 });
  }
}

/** @type {import('./$types').RequestHandler} */
export async function POST({ request }) {
  try {
    // ============================================================
    // ENDPOINT KHUSUS ADMIN UNTUK MEMBUAT USER
    // Hanya admin yang bisa membuat akun siswa/guru
    // ============================================================
    
    // TODO: Tambahkan middleware untuk verifikasi admin
    // Untuk sekarang, endpoint ini hanya bisa diakses dari halaman admin
    // yang sudah ter-authentikasi
    
    const data = await request.json();
    const { fullName, email, phone, role, major, password, nis, nip, address, class: className, gender, religion, subjects } = data;

    // =====================
    // VALIDASI
    // =====================
    if (!fullName || !email || !phone) {
      return json(
        { error: 'Nama, Email, dan Telepon wajib diisi' },
        { status: 400 }
      );
    }

    if (role === 'siswa' && !major) {
      return json(
        { error: 'Pilih jurusan untuk siswa' },
        { status: 400 }
      );
    }

    // Password login = NIS (siswa) atau NIP (guru). Jika admin isi password custom, bisa dipakai.
    const hasNis = role === 'siswa' && nis != null && String(nis).trim() !== '';
    const hasNip = role === 'guru' && nip != null && String(nip).trim() !== '';
    if (!hasNis && !hasNip && (!password || String(password).trim().length < 6)) {
      return json(
        { error: 'Isi NIS (siswa) atau NIP (guru) sebagai password login, atau password minimal 6 karakter' },
        { status: 400 }
      );
    }

    // =====================
    // CEK EMAIL & NIS DUPLIKAT
    // =====================
    const [existingUsers] = await db.query(
      'SELECT id FROM users WHERE email = ?',
      [email.trim()]
    );
    if (existingUsers.length > 0) {
      return json(
        { error: 'Email sudah terdaftar. Gunakan email lain.' },
        { status: 400 }
      );
    }

    if (role === 'siswa' && nis) {
      const [existingNis] = await db.query(
        'SELECT id FROM students WHERE nis = ?',
        [String(nis).trim()]
      );
      if (existingNis.length > 0) {
        return json(
          { error: 'NIS sudah terdaftar. Gunakan NIS lain.' },
          { status: 400 }
        );
      }
    }

    // =====================
    // GENERATE NIS / NIP (jika tidak disediakan)
    // =====================
    let nisOrNip = '';
    if (role === 'siswa') {
      const nisVal = nis != null && String(nis).trim() !== '' ? String(nis).trim() : null;
      nisOrNip = nisVal || (() => {
        const year = new Date().getFullYear().toString().slice(-2);
        const random = Math.floor(Math.random() * 1000)
          .toString()
          .padStart(3, '0');
        return year + random;
      })();
    } else if (role === 'guru') {
      nisOrNip = nip || (() => {
        return '19650101' +
          Math.floor(Math.random() * 1000000)
            .toString()
            .padStart(6, '0');
      })();
    }

    // =====================
    // PASSWORD LOGIN = NIS (siswa) / NIP (guru) yang didaftarkan admin
    // =====================
    const loginPassword = (role === 'siswa' ? nisOrNip : nisOrNip) || (password && password.trim()) || '';
    const hashedPassword = await bcrypt.hash(String(loginPassword).trim(), 10);

    // =====================
    // INSERT USERS + STUDENTS (siswa) dalam transaksi agar users & students selalu sinkron
    // =====================
    let userId;
    const conn = await db.getConnection();

    try {
      await conn.beginTransaction();

      const [userResult] = await conn.query(
        `INSERT INTO users
          (fullName, email, phone, role, password, major, status, createdAt)
         VALUES (?, ?, ?, ?, ?, ?, 'aktif', NOW())`,
        [
          fullName.trim(),
          email.trim(),
          phone.trim(),
          role,
          hashedPassword,
          role === 'siswa' ? major : null
        ]
      );
      userId = userResult.insertId;

      if (role === 'siswa') {
        const majorCode = (major || '').toUpperCase();
        const finalClassName = (className && String(className).trim()) || `X ${majorCode} 1`;

        // Kolom "class" di MySQL reserved word, pakai backtick
        await conn.query(
          `INSERT INTO students
            (nis, fullName, email, phone, address, \`class\`, major, gender, religion, status, createdAt)
           VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'aktif', NOW())`,
          [
            nisOrNip,
            fullName.trim(),
            email.trim(),
            phone.trim(),
            address && String(address).trim() ? address.trim() : 'Alamat belum diisi',
            finalClassName,
            major || null,
            gender && String(gender).trim() ? gender.trim() : null,
            religion && String(religion).trim() ? religion.trim() : null
          ]
        );

        // AUTO-CREATE/UPDATE CLASS (best-effort di dalam transaksi)
        try {
          const [existingClasses] = await conn.query(
            'SELECT * FROM classes WHERE name = ?',
            [finalClassName]
          );
          const [studentsInClass] = await conn.query(
            'SELECT COUNT(*) as count FROM students WHERE `class` = ? AND status = "aktif"',
            [finalClassName]
          );
          const studentCount = studentsInClass[0]?.count ?? 1;
          if (existingClasses.length > 0) {
            await conn.query(
              'UPDATE classes SET studentCount = ? WHERE name = ?',
              [studentCount, finalClassName]
            );
          } else {
            await conn.query(
              `INSERT INTO classes (name, major, homeroomTeacher, studentCount, status)
               VALUES (?, ?, '', ?, 'aktif')`,
              [finalClassName, major || '', studentCount]
            );
          }
        } catch (classErr) {
          console.error('⚠️ Error auto-create/update class:', classErr);
        }
      }

      await conn.commit();
    } catch (txError) {
      await conn.rollback();
      console.error('Admin create user/student transaction error:', txError?.message || txError);
      console.error('  code:', txError?.code, 'sqlMessage:', txError?.sqlMessage);
      throw txError;
    } finally {
      conn.release();
    }

    // =====================
    // INSERT TEACHERS
    // =====================
    if (role === 'guru') {
      // Convert subjects to JSON string
      let subjectsValue = '[]';
      console.log('📚 Subjects data received:', data.subjects, 'Type:', typeof data.subjects);
      
      if (data.subjects) {
        if (Array.isArray(data.subjects)) {
          subjectsValue = JSON.stringify(data.subjects);
          console.log('✅ Subjects is array, converted to JSON:', subjectsValue);
        } else if (typeof data.subjects === 'string' && data.subjects.trim() !== '') {
          // If it's a comma-separated string, convert to array then JSON
          const subjectsArray = data.subjects.split(',').map(s => s.trim()).filter(Boolean);
          subjectsValue = JSON.stringify(subjectsArray);
          console.log('✅ Subjects is string, converted to array then JSON:', subjectsValue);
        }
      } else {
        console.log('⚠️ No subjects provided, using empty array');
      }
      
      await db.query(
        `INSERT INTO teachers
          (nip, fullName, email, phone, subjects, class, status, createdAt)
         VALUES (?, ?, ?, ?, ?, ?, 'aktif', NOW())`,
        [nisOrNip, fullName.trim(), email.trim(), phone.trim(), subjectsValue, className || '']
      );
      
      console.log('✅ Teacher inserted with subjects:', subjectsValue);
    }

    // =====================
    // RESPONSE SUKSES ✅
    // =====================
    return json({
      success: true,
      message: `Akun ${role === 'siswa' ? 'siswa' : 'guru'} berhasil dibuat oleh admin.`,
      user: {
        id: userId,
        fullName,
        email,
        role,
        nis: role === 'siswa' ? nisOrNip : null,
        nip: role === 'guru' ? nisOrNip : null
      }
    });

  } catch (error) {
    console.error('Admin create user API error:', error?.message || error);
    console.error('  code:', error?.code, 'sqlMessage:', error?.sqlMessage);

    // Duplicate key: beri pesan yang jelas
    if (error?.code === 'ER_DUP_ENTRY' || error?.errno === 1062) {
      const msg = (error?.sqlMessage || '').toLowerCase();
      if (msg.includes('nis')) {
        return json({ error: 'NIS sudah terdaftar. Gunakan NIS lain.' }, { status: 400 });
      }
      if (msg.includes('email')) {
        return json({ error: 'Email sudah terdaftar. Gunakan email lain.' }, { status: 400 });
      }
      return json({ error: 'Data duplikat. NIS atau email mungkin sudah terdaftar.' }, { status: 400 });
    }

    // Koneksi / DB error
    if (error?.code === 'ECONNREFUSED' || error?.code === 'ER_ACCESS_DENIED_ERROR') {
      return json(
        { error: 'Database tidak dapat diakses. Periksa koneksi MySQL.' },
        { status: 503 }
      );
    }

    return json(
      { error: 'Terjadi kesalahan pada server. Silakan coba lagi.' },
      { status: 500 }
    );
  }
}
